(function($) {
    $.fn.fileUploadPlugin = function(options) {
        // Default settings
        var settings = $.extend({
            allowedFileTypes: ['image/jpeg', 'image/png', 'image/gif'],
            maxFileSize: 5 * 1024 * 1024, // 5MB
            previewContainer: '.preview-container',
            dropArea: '.drop-area',
            uploadButton: '#upload-btn',
            uploadUrl: '/upload', // Your server URL for handling uploads
            borderColor: '#C0C0C0' // Default border color for the drop area
        }, options);

        // CSS Styles directly in the plugin
        var styles = `
            .drop-area {
                border: 2px dashed ${settings.borderColor};
                padding: 20px;
                text-align: center;
                margin-bottom: 20px;
                width: 100%;
            }

            .preview-container {
                display: flex;
                flex-wrap: wrap;
                justify-content: flex-start;
            }

            .preview-item {
                position: relative;
                margin: 10px;
                max-width: 150px;
                text-align: center;
                border: 1px solid #ddd;
                padding: 10px;
                box-sizing: border-box;
            }

            .preview-item img {
                width: 100%;
                height: auto;
            }

            .preview-item button {
                position: absolute;
                top: 5px;
                right: 5px;
                background-color: rgba(0, 0, 0, 0.5);
                color: white;
                border: none;
                cursor: pointer;
                border-radius: 50%;
            }

            .dragover {
                background-color: #f0f0f0;
            }
        `;

        // Inject styles into the document head
        $('head').append('<style>' + styles + '</style>');

        // Handle file uploads (including drag and drop)
        this.each(function() {
            var $container = $(this);

            // Create an invisible file input and trigger it when clicking anywhere in the container
            var $fileInput = $('<input type="file" multiple style="display:none;" />');
            $container.append($fileInput);

            // Trigger file input dialog when clicking anywhere in the plugin container
            $container.on('click', function(event) {
                if (!$(event.target).closest($fileInput).length) {
                    // Open the file input dialog on clicking the container (except the file input itself)
                    $fileInput.click();
                }
            });

            // Handle file input change (triggered by clicking or drag-and-drop)
            $fileInput.on('change', function(event) {
                handleFiles(event.target.files, $container);
            });

            // Handle drag over and drop events for the drop area
            $container.find(settings.dropArea).on('dragover', function(e) {
                e.preventDefault();
                $(this).addClass('dragover');
            });

            $container.find(settings.dropArea).on('dragleave', function(e) {
                e.preventDefault();
                $(this).removeClass('dragover');
            });

            $container.find(settings.dropArea).on('drop', function(e) {
                e.preventDefault();
                $(this).removeClass('dragover');
                handleFiles(e.originalEvent.dataTransfer.files, $container);
            });

            // Handle file preview, validation, and removal
            function handleFiles(files, $container) {
                for (var i = 0; i < files.length; i++) {
                    var file = files[i];

                    // Validate file type and size
                    if (!settings.allowedFileTypes.includes(file.type)) {
                        alert('Invalid file type: ' + file.name);
                        continue;
                    }

                    if (file.size > settings.maxFileSize) {
                        alert('File size exceeds the limit: ' + file.name);
                        continue;
                    }

                    // Create file preview
                    var reader = new FileReader();
                    reader.onload = function(e) {
                        var previewItem = $('<div class="preview-item"></div>');
                        previewItem.append('<img src="' + e.target.result + '" alt="Preview" />');
                        previewItem.append('<button class="remove-file">X</button>');

                        // Add preview item to the container
                        $container.find(settings.previewContainer).append(previewItem);
                    };
                    reader.readAsDataURL(file);
                }
            }

            // Enable reordering of previews using jQuery UI sortable
            $container.find(settings.previewContainer).sortable({
                items: '.preview-item',
                cursor: 'move'
            });

            // Remove file preview on button click
            $container.find(settings.previewContainer).on('click', '.remove-file', function() {
                $(this).parent().remove();
            });

            // Handle the upload button click (for uploading all files in the sequence)
            $container.find(settings.uploadButton).on('click', function() {
                uploadFiles($container);
            });

            // Upload files to the server
            function uploadFiles($container) {
                var formData = new FormData();

                // Collect all files from the preview container (files that are displayed)
                var files = $container.find(settings.previewContainer).find('img').map(function() {
                    return $(this).attr('src');
                }).get();

                // Send files to the server using AJAX
                $.ajax({
                    url: settings.uploadUrl,
                    type: 'POST',
                    data: formData,
                    processData: false,
                    contentType: false,
                    success: function(response) {
                        alert('Files uploaded successfully!');
                    },
                    error: function(error) {
                        alert('Upload failed!');
                    }
                });
            }
        });

        return this;
    };
}(jQuery));
